package easik.sketch.attribute;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;

import easik.Easik;
import easik.sketch.vertex.EntityNode;


/**
 * Displays a UI for creating/editing a unique key
 * 
 * @author Kevin Green 2006
 * @since 2006-06-05 Kevin Green
 * @version 2006-06-08 Kevin Green
 */
public class UniqueKeyUI  extends JDialog{

	/**
	 * The unique key being formed in this UI
	 */
	private UniqueKey _uniqueKey;
	/**
	 * The name of the unique key
	 */
	private String _keyName;
	/**
	 * The list of attributes in the associated Entity node
	 */
	private ArrayList _attList;
	/**
	 * The current entity node
	 */
	private EntityNode _entity;
	/**
	 * Stores whether this is a new Unique key, or editing of an old one.
	 */
	private boolean _isNew;
	/**
	 * Stores whether the user has selected to exit the UI.  Initially set to <b>false</b>
	 */
	private boolean _exit = false;

	/**
	 * The JTextField used to store the key name
	 */
	private JTextField _keyNameField;
	/**
	 * The JList of attributes
	 */
	private JList _attListField;
	/**
	 * The JScrollPane used to hold the list of attributes
	 */
	private JScrollPane _attListPane;

	/**
	 * The ok button
	 */
	private JButton _okButton;
	/**
	 * The cancel button
	 */
	private JButton _cancelButton;

	/**
	 * The window length
	 */
	private static final int WINDOW_LENGTH = 400;
	/**
	 * The window height
	 */
	private static final int WINDOW_HEIGHT = 200;
	/**
	 * The window's x-position
	 */
	private static final int WINDOW_POS_X = 300;
	/**
	 * The window's y-position
	 */
	private static final int WINDOW_POS_Y = 250;

	/**
	 * Creates a dialog box prompting the user for input.
	 * Used for creation of new unique key.
	 * @param inEntity The Entity node for which the user is creating a unique key
	 */
	public UniqueKeyUI(EntityNode inEntity){
		super();
		this.setSize(WINDOW_LENGTH, WINDOW_HEIGHT);
		this.setResizable(false);
		this.setTitle("Attribute Values");
		this.setBounds(WINDOW_POS_X, WINDOW_POS_Y, WINDOW_LENGTH, WINDOW_HEIGHT);
		
		_entity = inEntity;
		
		_isNew = true;
		
		createDialogBox();
		
		this.setModal(true);
		this.setVisible(true);
	}
	
	/**
	 * Creates a dialog box prompting the user for input.
	 * Used for editing of unique key.
	 * 
	 * @param inEntity The EntityNode for which a unique key is being built
	 * @param inKey The current unique key being edited
	 */
	public UniqueKeyUI(EntityNode inEntity, UniqueKey inKey){
		super();
		this.setSize(WINDOW_LENGTH, WINDOW_HEIGHT);
		this.setResizable(false);
		this.setTitle("Attribute Values");
		this.setBounds(WINDOW_POS_X, WINDOW_POS_Y, WINDOW_LENGTH, WINDOW_HEIGHT);
		
		_entity = inEntity;
		_uniqueKey = inKey;
		
		_isNew = false;
		
		createDialogBox();
		
		this.setModal(true);
		this.setVisible(true);
	}
	
	/**
	 * Creates a dialog box which prompts the user for attribute information
	 */
	private void createDialogBox(){
		
		JPanel mainPane = new JPanel(new BorderLayout());
		
		//Add the title
		JLabel title = new JLabel("     ", JLabel.CENTER);
		mainPane.add(title, BorderLayout.NORTH);
		
		//Create info pane
		JPanel attInfoPane = new JPanel(new GridLayout(5, 1));
		
		//Add the key name text box
		attInfoPane.add(new JLabel("Key Name:"));
		_keyNameField = new JTextField();
		attInfoPane.add(_keyNameField);
		
		//Add the attributes box
		attInfoPane.add(new JLabel("Attribute Included in Key:"));
		_attListField = new JList(_entity.getAttributes().toArray());
		_attListPane = new JScrollPane(_attListField);
		attInfoPane.add(_attListPane);
		
		if(!_isNew){
			Object[] atts = _entity.getAttributes().toArray();
			ArrayList keyAtts = _uniqueKey.getAttributes();
			int[] keyAttsIndex = new int[keyAtts.size()];
			for(int i=0; i<keyAtts.size(); i++){
				for(int j=0; j<atts.length; j++){
					if(atts[j].equals(keyAtts.get(i))){
						keyAttsIndex[i]=j;
					}
				}
			}
			_attListField.setSelectedIndices(keyAttsIndex);
			_keyNameField.setText(_uniqueKey.getKeyName());
		}
		
		//Add info pane to main pane
		mainPane.add(attInfoPane, BorderLayout.CENTER);
		
		//Add the buttons
		if(_isNew){
			_okButton = new JButton("Insert");
		}
		else{
			_okButton = new JButton("Update");
		}
		_okButton.setSelected(true);
		_cancelButton = new JButton("Cancel");
		
		JPanel buttons = new JPanel(new GridLayout(1, 2));
		buttons.add(_okButton);
		buttons.add(_cancelButton);
		
		mainPane.add(buttons, BorderLayout.SOUTH);
		mainPane.setVisible(true);
		
		//Padding to make main pane look better
		mainPane.add(new JLabel("              "), BorderLayout.WEST);
		mainPane.add(new JLabel("              "), BorderLayout.EAST);
		
		this.getContentPane().add(mainPane);
		
		//Create action listeners for buttons
		_okButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				okPressed();
			}
		});
		
		_cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				cancelPressed();
			}
		});
	}
	
	/**
	 * Method called when the ok button is pressed.
	 */
	public void okPressed(){
		_keyName = _keyNameField.getText();
		if(_keyName.equals("")){
			JOptionPane.showMessageDialog(Easik.getInstance().getFrame(),
					"You have entered an empty key name.\nPlease try again.",
					"Empty Key Name", JOptionPane.ERROR_MESSAGE);
			return;
		}
		
		_attList = new ArrayList();
		Object[] tempAttList = _attListField.getSelectedValues();
		for(int i=0; i<tempAttList.length; i++){
			_attList.add(tempAttList[i]);
		}
		
		_exit = true;
		dispose();
	}
	
	/**
	 * Method called when the cancel button is pressed.
	 */
	public void cancelPressed(){
		System.out.println("Cancel Pressed");
		dispose();
	}
	
	/**
	 * Returns a list of the attributes in the key
	 * 
	 * @return A list of the attributes in the key
	 */
	public ArrayList getAttList(){
		return _attList;
	}
	
	/**
	 * Returns the name of the key
	 * 
	 * @return The name of the key
	 */
	public String getKeyName(){
		return _keyName;
	}
	
	/**
	 * Returns the exit value
	 * 
	 * @return The exit value
	 */
	public boolean getExit(){
		return _exit;
	}
	
}
